{ *********************************************************************** }
{                                                                         }
{ Delphi Visual Component Library                                         }
{                                                                         }
{ Copyright (c) 2003-2004 Borland Software Corporation                    }
{                                                                         }
{ *********************************************************************** }

unit Borland.Vcl.WinFormContainer platform experimental;

{$T-,H+,X+}

interface

uses
  Windows, Messages, Classes, Controls, System.Windows.Forms;

{ TWinFormContainer hosts a System.Windows.Forms.Control.
  This allows the WinForms control to appear as the child of a VCL windowdowed
  control. If you change the size, position, visibility, or enabled state of
  either the contained control or the TWinFormContainer, they are both
  updated to stay in sync (for example, if you move the contained control, it
  is the container that actually moves, not the contained control. There is no
  attempt to synchronize any other properties (such as font, color, etc). To
  change these, use the properties of the contained control. You can access the
  contained control using the Control property. }
type
  TWinFormContainer = class(TWinControl)
  private
    FControl: System.Windows.Forms.Control;
    FSynchronizingSize: Boolean;
    FSynchronizingEnabled: Boolean;
    FSynchronizingVisible: Boolean;
    procedure SetControl(AControl: System.Windows.Forms.Control);
    procedure UpdateControlParent;
  protected
    procedure Resize; override;
    procedure ChangeScale(M, D: Integer); override;
    procedure CreateWnd; override;
    procedure WFCResize(Sender: System.Object; AArgs: System.EventArgs);
    procedure WFCEnabledChanged(Sender: System.Object; AArgs: System.EventArgs);
    procedure WFCVisibleChanged(Sender: System.Object; AArgs: System.EventArgs);
  public
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure CMVisibleChanged(var Message: TMessage); message CM_VISIBLECHANGED;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    property Control: System.Windows.Forms.Control read FControl write SetControl;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Align;
    property Anchors;
    property DragKind;
    property DragCursor;
    property DragMode;
    property PopupMenu;
    property Visible;
    property OnAlignInsertBefore;
    property OnAlignPosition;
    property OnDockDrop;
    property OnDockOver;
    property OnEnter;
    property OnExit;
    property OnGetSiteInfo;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnUnDock;

    property OnCanResize;
    property OnClick;
    property OnConstrainedResize;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnMouseActivate;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnMouseWheel;
    property OnMouseWheelDown;
    property OnMouseWheelUp;
    property OnResize;
    property OnStartDock;
    property OnStartDrag;

    property BevelEdges;
    property BevelInner;
    property BevelOuter;
    property BevelKind;
    property BevelWidth;
    property BorderWidth;
  end;

{AddWinFormsControl adds a System.Windows.Forms.Control instance as the child
 of a VCL windowed control. It returns the TWinFormContainer that is
 created to adapt the WinForms control to the VCL parent. }
function AddWinFormsControl(Owner: TComponent; Ctrl: System.Windows.Forms.Control;
  AParent: TWinControl; X, Y: Integer): TWinFormContainer;

// Same as AddWinFormsControl only
function AddWinFormsControlAndParent(Parent: TWinControl;
  Ctrl: System.Windows.Forms.Control; X, Y: Integer): TWinFormContainer;

{NewWinFormsControl creates a System.Windows.Forms.Control instance of a specified
 type and adds it as the child of a VCL windowed control. It returns the
 TWinFormContainer that is created to adapt the WinForms control to the
 VCL parent. Params is an array of parameters for the constructor of the
 specified child control. Typically, Params has length 0. }
function NewWinFormsControl(Owner: TComponent; CtrlType: System.Type;
  Params: array of TObject; AParent: TWinControl; X, Y: Integer): TWinFormContainer;

implementation

uses
  SysUtils, Forms, RTLConsts, System.Reflection;

function  AddWinFormsControl(Owner: TComponent; Ctrl: System.Windows.Forms.Control;
  AParent: TWinControl; X, Y: Integer): TWinFormContainer;
begin
  Result := TWinFormContainer.Create(Owner);
  Result.Parent := AParent;
  Result.Left := X;
  Result.Top := Y;
  Result.Control := Ctrl;
  Result.Invalidate;
end;

function  AddWinFormsControlAndParent(Parent: TWinControl;
  Ctrl: System.Windows.Forms.Control; X, Y: Integer): TWinFormContainer;
begin
  Result := TWinFormContainer.Create(Parent);
  Result.Parent := Parent;
  Result.Left := X;
  Result.Top := Y;
  Result.Control := Ctrl;
  Result.Invalidate;
end;

function  NewWinFormsControl(Owner: TComponent; CtrlType: System.Type;
  Params: array of TObject; AParent: TWinControl; X, Y: Integer): TWinFormContainer;
var
  I: Integer;
  ParamTypes: array of System.Type;
  CInfo: System.Reflection.ConstructorInfo;
begin
  SetLength(ParamTypes, Length(Params));
  for I := 0 to Length(Params) - 1 do
    ParamTypes[I] := Params[I].GetType;
  CInfo := CtrlType.GetConstructor(ParamTypes);
  if CInfo = nil then
    raise Exception.CreateFmt(SNoConstructorFound, [CtrlType.ToString]);
  Result := AddWinFormsControl(Owner,
    System.Windows.Forms.Control(CInfo.Invoke(Params)), AParent, X, Y);
end;

{ TWinFormContainer }

constructor TWinFormContainer.Create(AOwner: TComponent);
begin
  inherited;
  ControlStyle := ControlStyle + [csAcceptsControls];
end;

destructor TWinFormContainer.Destroy;
begin
  FControl := nil;
  inherited;
end;

procedure TWinFormContainer.UpdateControlParent;
begin
  if not FControl.IsHandleCreated then
    FControl.CreateControl;
  Windows.SetParent(HWND(FControl.Handle), Handle);
  FControl.Invalidate;
end;

procedure TWinFormContainer.SetControl(AControl: System.Windows.Forms.Control);
begin
  if FControl = AControl then Exit;
  if AControl <> nil then
  begin
    FControl := AControl;
    SetBounds(Left, Top, AControl.Width, AControl.Height);
    AControl.SetBounds(0, 0, AControl.Width, AControl.Height);
    Enabled := AControl.Enabled;
    Visible := AControl.Visible;
    AControl.add_LocationChanged(@WFCResize);
    AControl.add_SizeChanged(@WFCResize);
    AControl.add_EnabledChanged(@WFCEnabledChanged);
    AControl.add_VisibleChanged(@WFCVisibleChanged);
    UpdateControlParent;
  end else
  begin
    FControl.remove_LocationChanged(@WFCResize);
    FControl.remove_SizeChanged(@WFCResize);
    FControl.remove_EnabledChanged(@WFCEnabledChanged);
    FControl.remove_VisibleChanged(@WFCVisibleChanged);
    FControl := AControl;
  end;
end;

procedure TWinFormContainer.CreateWnd;
begin
  inherited;
  if Assigned(FControl) then
    UpdateControlParent;
end;

procedure TWinFormContainer.ChangeScale(M, D: Integer);
begin
  if FControl = nil then
    inherited
  else if M <> D then
     FControl.Scale(M/D);
   // do we need to call the inherited? or does the resize come back from FControl?
end;

procedure TWinFormContainer.CMEnabledChanged(var Message: TMessage);
begin
  if (not FSynchronizingEnabled) and (FControl <> nil) then
  begin
    FSynchronizingEnabled := True;
    try
      FControl.Enabled := Enabled;
    finally
      FSynchronizingEnabled := False;
    end;
  end;
  inherited;
end;

procedure TWinFormContainer.WFCEnabledChanged(Sender: System.Object; AArgs: System.EventArgs);
begin
  if not FSynchronizingEnabled then
  begin
    FSynchronizingEnabled := True;
    try
      Enabled := FControl.Enabled;
    finally
      FSynchronizingEnabled := False;
    end;
  end;
end;

procedure TWinFormContainer.CMVisibleChanged(var Message: TMessage);
begin
  if (not FSynchronizingVisible) and (FControl <> nil) then
  begin
    FSynchronizingVisible := True;
    try
      FControl.Visible := Visible;
    finally
      FSynchronizingVisible := False;
    end;
  end;
  inherited;
end;

procedure TWinFormContainer.WFCVisibleChanged(Sender: System.Object; AArgs: System.EventArgs);
begin
  if not FSynchronizingVisible then
  begin
    FSynchronizingVisible := True;
    try
      Visible := FControl.Visible;
    finally
      FSynchronizingVisible := False;
    end;
  end;
end;

procedure TWinFormContainer.Resize;
begin
  inherited;
  if not (not FSynchronizingSize) and (FControl <> nil) then
  begin
    FSynchronizingSize := True;
    try
      FControl.SetBounds(0, 0, ClientWidth, ClientHeight);
    finally
      FSynchronizingSize := False;
    end;
  end;
end;

procedure TWinFormContainer.WFCResize(Sender: System.Object; AArgs: System.EventArgs);
begin
  if not FSynchronizingSize then
  begin
    FSynchronizingSize := True;
    try
      SetBounds(Left + FControl.Left, Top + FControl.Top, FControl.Width, FControl.Height);
      FControl.SetBounds(0,0, Width, Height);
    finally
      FSynchronizingSize := False;
    end;
  end;
end;

procedure TWinFormContainer.WMPaint(var Message: TWMPaint);
var
  Msg: System.Windows.Forms.Message;
begin
  if (Message.DC <> 0) and (FControl <> nil) then
  begin
    Msg := System.Windows.Forms.Message.&Create(FControl.Handle,
      WM_PRINT, IntPtr(Integer(Message.DC)),
      IntPtr(Integer(PRF_NONCLIENT or PRF_CLIENT or PRF_CHILDREN)));
    FControl.WindowTarget.OnMessage(Msg);
  end
  else
    inherited;
end;

end.

